﻿/*
VERSION: 1.2
1.2		Renamed internal "change_radian" event to "change".  Player still receives "change_radian" event
1.1		The radian wraps between 0 and PI*2  (The radian equivalent of 360 degrees)

USAGE:
	#include "functions/tank_turn.as"
	player_mc.turn_obj = makeTankTurn( player_mc );											// this handles rotation of the "radian"
	#include "functions/tank_move.as"
	player_mc.move_obj = makeTankMove( player_mc, player_mc.sprite );		// this moves the player
	
	player_mc.turn_obj.resetVelocity();		// set velocity to Zero		(useful when switching between movement systems in real-time)
	player_mc.move_obj.resetVelocity();		// set velocity to Zero		(useful when switching between movement systems in real-time)
	
	// tank_turn will emit "change_radian" events within player_mc
	// tank_move will detect those events within player_mc and update its own radian
	
	onUnload = function(){
		clearInterval(turn_obj.loopInterval);
		clearInterval(move_obj.loopInterval);
	}
	
	
SETTINGS:
	turn_maxVel		Maximum rotation speed  (in radians)
	turn_acel			Rotation acelleration,  how many radians to rotate per frame while a key is pressed
	turn_decel		Rotation deceleration, multiplier that reduces velocity per frame when no key is pressed
	
	key_turnLeft			Key code for rotating left
	key_turnRight			Key code for rotating right
	
	
PROPERTIES:
	maxSpeed			Maximum rotation speed  (in radians)
	acel					Acelleration speed,  how many pixels the velocity increases per frame while a key is pressed
	decel					Deceleration speed, multiplier that reduces velocity per frame when no key is pressed
	radian				Current rotation,  used to represent the facing direction
	

RAM.keys defaults
	RAM.keys.left
	RAM.keys.right
	
	
EVENTS: 
	unload
	change					Same as change_radian
	
PLAYER EVENTS: 
	change_radian		{value: Point}		(local-relative movement.  -y forward  +y backward  -x left  +x right)			Only fires within player
	
RESPONDS TO PLAYER EVENTS:
	request_radian
	change_radian
	unload
*/
function makeTankTurn( player_mc ){
	#include "eventSystem3.as"
	var _this = {};		// external interface
	AsBroadcaster.initialize( _this );
	var react = make_react( _this );		// param is optional
	react.to("unload").from( player_mc ).tell( _this );
	react.to("request_radian").from( player_mc ).tell( _this );
	react.to("change").tell( player_mc, "change_radian");
	
	
	
	// Variables
	var fps = 30;
	// movement speed
	_this.turn_acel = Math.PI /180 *3;				// acelleration increment		(3 degrees of acelleration)
	_this.turn_decel = 0.6;				// brake multiplier
	_this.turn_vel = 0;
	_this.turn_maxVel = Math.PI /180 *14;		// 14 degrees of rotation
	_this.turn_radian = Math.PI /180 *90;		// current rotation  (in radians)
	// controls
	_this.key_turnLeft = RAM.keys.left || Key.LEFT;
	_this.key_turnRight = RAM.keys.right || Key.RIGHT;
	// allow the "radian" rotation to be accessed externally
	_this.addProperty( "radian", function(){
		return _this.turn_radian
	}, function( newValue ){
		// ignore same values
		if( _this.turn_radian === newValue )		return;
		
		_this.turn_radian = newValue;
		sendEvent( "change", {value: _this.turn_radian}, _this);
		// sendEvent( "change_radian", {value: _this.turn_radian}, player_mc);
	});
	
	// legacy
	_this.addProperty( "acel", function(){
		return _this.turn_acel
	}, function( newValue ){
		_this.turn_acel = newValue;
	});
	_this.addProperty( "decel", function(){
		return _this.turn_decel
	}, function( newValue ){
		_this.turn_decel = newValue;
	});
	_this.addProperty( "maxSpeed", function(){
		return _this.turn_maxVel
	}, function( newValue ){
		_this.turn_maxVel = newValue;
	});
	
	// Data request events
	react.to("request_radian").then = function( evt ){
		evt.callback( _this.turn_radian );
	}
	
	// update radian when external code changes it
	react.to("change_radian").from( player_mc ).then = function( evt ){
		_this.radian = Number( evt.value ) || 0;
	}
	
	
	
	_this.resetVelocity = function(){
		_this.turn_vel = 0;
	}// resetVelocity()
	
	
	
	
	// // // // // // // // // // // // // // // // // // // // // // // // // // // // // // 
	// Main Program
	var max_radian = Math.PI * 2;
	_this.loop = function(){
		//  Abort if the player sprite doesn't exist
		if(player_mc._currentframe === undefined){
			sendEvent("unload");
			return;
		}// if:   player_mc is missing
		
		_this.turn_vel = changeTurnVelocity( _this.turn_vel );
		// update total speeds
		_this.turn_radian += _this.turn_vel;		// radian
		// wrap-around the values between 0 and max-radian  (360 degrees)
		while( _this.turn_radian < 0 )		_this.turn_radian += max_radian;
		_this.turn_radian %= max_radian;
		
		// announce rotation
		if(_this.turn_vel !== 0){
			sendEvent( "change", {value: _this.turn_radian}, _this);
			// sendEvent( "change_radian", {value: _this.turn_radian}, player_mc);
		}
	}// loop()
	_this.loopInterval = setInterval( function(){
		_this.loop();
	}, 1000/fps );
	react.to("unload").then = function(){
		clearInterval(_this.loopInterval);
	}
	
	return _this;
	
	
	
	
	
	// // // // // // // // // // // // // // // // // // // // // // // // // // // // // // 
	function changeTurnVelocity( turn_vel ){
		// turn
		if( _this.key_turnRight  &&  Key.isDown( _this.key_turnRight ) ){
			if( turn_vel < _this.turn_maxVel ){
				if(turn_vel < 0)		decellerateTurn();
				turn_vel += _this.turn_acel;
				// limit the max speed
				if( turn_vel > _this.turn_maxVel )		turn_vel = _this.turn_maxVel;
			}// if:   not at or beyond the max speed
		}else if( _this.key_turnLeft  &&  Key.isDown( _this.key_turnLeft ) ){
			if( turn_vel > -_this.turn_maxVel ){
				if(turn_vel > 0)		decellerateTurn();
				turn_vel -= _this.turn_acel;
				// limit the max speed
				if( turn_vel < -_this.turn_maxVel )		turn_vel = -_this.turn_maxVel;
			}// if:   not at or beyond the max speed
		}else{
			decellerateTurn();
		}
		function decellerateTurn(){
			turn_vel *= _this.turn_decel;
			if( Math.abs( turn_vel ) < 0.017 )		turn_vel = 0;		// if less than 1 degree of rotation,  then => no rotation
		}// decellerateTurn()
		
		return turn_vel;
	}// changeTurnVelocity()
	
}// makeTankTurn()